%% Demo script to test the AEM (Alternating Extragradient Method) on restoration problems
%% Data from .mat files
% This software is developed within the research project
%
%        PRISMA - Optimization methods and software for inverse problems
%                           http://www.unife.it/prisma
%
% funded by the Italian Ministry for University and Research (MIUR), under the
% PRIN2008 initiative, grant n. 2008T5KA4L, 2010-2012.
%
% Version: 2.0
% Date:    May 2012
%
%
% ------------------------------------------------------------------------------
% % Authors: 
%   Silvia Bonettini, Valeria Ruggiero, Alessandro Benfenati
%    Dept. of Mathematics, University of Ferrara, Italy
%    silvia.bonettini@unife.it, rgv@unife.it, bnflsn@unife.it
%
% Software homepage: http://www.unife.it/prisma/software
%
% Copyright (C) 2012 by S. Bonettini, V. Ruggiero, A. Benfenati.
% ------------------------------------------------------------------------------
% COPYRIGHT NOTIFICATION
%
% Permission to copy and modify this software and its documentation for 
% internal research use is granted, provided that this notice is retained 
% thereon and on all copies or modifications. The authors and their
% respective Universities makes no representations as to the suitability 
% and operability of this software for any purpose. It is provided "as is"
% without express or implied warranty. Use of this software for commercial
% purposes is expressly prohibited without contacting the authors.
%
% This program is free software; you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your 
% option) any later version.
%
% This program is distributed in the hope that it will be useful, but 
% WITHOUT ANY WARRANTY; without even the implied warranty of 
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
% See the GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License along 
% with this program; if not, either visite http://www.gnu.org/licenses/
% or write to
% Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
% =========================================================================
% 

clear all
close all
clc

addpath SideFuns
addpath RegFuns

%% define name of test problem
nameobj = 'fantoccio_1000';
% Load data
 [obj, gn, psf, bg, beta] = dataload(nameobj);
% [obj, gn, psf, bg, beta, sol] = dataload(nameobj);

% obj is the original image, gn is the detected image, psf is the image matrix of
% the Point Spread Function, bg is the backgroud term, beta is the
% regularization parameter.
% In case of unknown true object and unknown optimal beta, these 2 variables
% are set as void [] in the .mat files;

%% select 'denoising or 'deblurring'
problem = 'deblurring';
TF = set_opt(psf, problem);

if isempty(obj) 
    bench = {}; 
else
    bench = {obj}; % Cell array containing an arbitrary number of images to
                   % be compared with the reconstruction computed at each
                   % iteration.
end

if isempty(beta)
    beta = 0.01; % Attempt value for the reg. parameter (in case of unknown optimal value); 
end

savedata = 0; % saving option

%% Definition of anonymous functions for dicrete convolution operators with 
%% the PSF (A) and with its transpose (AT)
% function handles for matrix--vector product Ax and transpose matrix-vector 
% product A^T x; here A is the matrix representing the distorsion due to 
% the acquisition system; 
A   = @(x) Afunction(x,TF,size(gn));
AT  = @(x) Afunction(x,conj(TF),size(gn));

nonzeroindex  = gn > 0; 
zeroindex     = ~nonzeroindex;
%% Setting the lower bounds for the solution, according to the paper
% S. BONETTINI, V. RUGGIERO: " An Alternating Extragradient Method for Total Variation 
% based image restoration from Poisson data", Inverse Problems, 27, 2011.
eta = zeros(size(gn));
if strcmp(problem,'denoising')
    eta(nonzeroindex) = min(gn(nonzeroindex));
end

%% starting point
x_initial = max(eta,gn);

%% AEM call
[x, TimeCost, InnVec, Primal, phi_xy, alpha_vec, err, KL_vec] = ...
    AEM_MD(gn, A, AT, beta,...
    'OBJ', bench, ...
    'X_INITIAL', x_initial, ...
    'MAXIT', 30000, ...
    'STOPCRITERIUM', 1, ...
    'VERBOSE',1, ...
    'ETA', eta, ...
    'EPSILON', 1e-8, ...
    'PHI_PAR', 0, ...
    'ERM', 5);

% The resulting image is stored in x

%% Save data
if savedata
    nameobjs = [nameobj,'_AEM'];
    datasave
end

% ==============================================================================
% End of main_AEM.m file 
% ==============================================================================

