function [x, iter, err, discr, times] = sgp_deblurring(A, gn, varargin)
% sgp_deblurring - SGP algorithm for non-regularized deblurring
%   This function solves an image deblurring problem by applying the SGP
%   algorithm to the minimization of the generalized Kullback-Leibler  
%   divergence with no regularization [1]:
%
%      min KL(A*x + bg, gn)
%       x in OMEGA
%
%   where KL(u,v) is the generalized Kullback-Leibler divergence between 
%   vectors u and v, bg is the background, gn are the observed data and 
%   the feasible set OMEGA is either
%   'nonneg':    x(i) >= 0;
%   'nonneg_eq': same as 'nonneg' plus a liner constraint of total flux
%                conservation.
%
%   Note: the PSF must be normalized. All columns of A must sum-up to 1.
%
%   [1] S. Bonettini, R. Zanella, L. Zanni, 
%       "A scaled gradient projection method for constrained image deblurring", 
%       Inverse Problems 25(1), 2009, January, 015002.
%
% SYNOPSIS
%   [x, iter, err, discr, time] = sgp_deblurring(A, gn[, opts])
%
% MANDATORY INPUT
%   A   (double array or func handle) 
%                      - measuring matrix or function handle used to apply 
%                        the blurring operator, that is to compute A*x
%                        If A is a function handle, also AT is required.
%                        N.B. All columns of A must sum-up to 1.
%   gn  (double array) - measured image
%
% OPTIONAL INPUT
%   The following options must be provided as keyword/value pairs.
%   'AT'               - Function handle to compute transpose(A)*x
%   'OBJ'              - Exact solution, for error calculation (double array)
%   'BG'               - Background value (double)
%                        DEFAULT = 0
%   'INITIALIZATION'   - Choice for starting point:
%                        0  - all zero starting point
%                        1  - random starting point
%                        2  - initialization with gn
%                        3  - initialization with
%                               ones(size(gn))*sum(gn(:) - bg) / numel(gn)
%                        x0 - user-provided starting point (double array)
%                        DEFAULT = 0
%   'MAXIT'            - Maximum number of iterations (integer)
%                        DEFAULT = 1000
%   'VERBOSE'          - Verbosity level (integer)
%                        0 - silent
%                        1 - print configuration parameters at startup
%                        2 - print configuration parameters at startup and
%                            some information at each iteration
%                        DEFAULT = 0
%   'STOPCRITERION'    - Choice for stopping rule (integer)
%                        1 -> iter > MAXIT
%                        2 -> ||x_k - x_(k-1)|| <= tol*||x_k|| OR iter > MAXIT
%                        3 -> |KL_k - KL_(k-1)| <= tol*|KL_k| OR iter > MAXIT
%                        4 -> (2/N)*KL_k <= tol OR iter > MAXIT
%                        DEFAULT = 1;
%   'TOL'              - Tolerance used in the stopping criterion
%                        DEFAULT = 1e-4 if STOPCRITERION = 2 or 3
%                        DEFAULT = 1+1/mean(gn) if STOPCRITERION = 4
%   'M'                - Nonmonotone lineasearch memory (positive integer)
%                        If M = 1 the algorithm is monotone
%                        DEFAULT = 1 (monotone)
%   'GAMMA'            - Linesearch sufficient-decrease parameter (double)
%                        DEFAULT = 1e-4
%   'BETA'             - Linesearch backtracking parameter (double)
%                        DEFAULT = 0.4
%   'ALPHA_MIN'        - Lower bound for Barzilai-Borwein' steplength (double>0)
%                        DEFAULT = 1e-5
%   'ALPHA_MAX'        - upper bound for Barzilai-Borwein' steplength (double>0)
%                        DEFAULT = 1e5
%   'MALPHA'           - Memory length for alphaBB2 (positive integer)
%                        DEFAULT = 3
%   'TAUALPHA'         - Alternating parameter for Barzilai-Borwein' steplength
%                        (double)
%                        DEFAULT = 0.5
%   'INITALPHA'        - Initial value for Barzilai-Borwein' steplength (double)
%                        DEFAULT = 1.3
%   'OMEGA'            - Constraints type (string):
%                       'nonneg'   : non negativity
%                       'nonneg_eq': non negativity and flux conservation
%                                    ( estimated flux is sum(gn(:) - bg) )
%                        DEFAULT: 'nonneg'
%   'SAVE'             - Directory name: for each iteration saves:
%                        - the reconstructed image x_k
%                        - the Puetter residual: (x_k - gn) ./ sqrt(x_k)
%
% OUTPUT
%   x                  - Reconstructed data
%   iter               - Number of iterations
%   err                - Error value at each iteration. If OBJ was not given, 
%                        then err is the empty matrix.
%   discr              - Discrepancy value after each iteration:
%                            D = 2/numel(x_k) * KL( Ax_k + bg, gn)
%   time               - CPU time after each iteration
%
% ------------------------------------------------------------------------------
%
% This software is developed within the research project
%
%        PRISMA - Optimization methods and software for inverse problems
%                           http://www.unife.it/prisma
%
% funded by the Italian Ministry for University and Research (MIUR), under
% the PRIN2008 initiative, grant n. 2008T5KA4L, 2010-2012. This software is
% part of the package "IRMA - Image Reconstruction in Microscopy and Astronomy"
% currently under development within the PRISMA project.
%
% Version: 1.0
% Date:    July 2011

% Authors: 
%   Riccardo Zanella, Gaetano Zanghirati
%    Dept. of Mathematics, University of Ferrara, Italy
%    riccardo.zanella@unife.it, g.zanghirati@unife.it
%   Roberto Cavicchioli, Luca Zanni
%    Dept. of Pure Appl. Math., Univ. of Modena and Reggio Emilia, Italy
%    roberto.cavicchioli@unimore.it, luca.zanni@unimore.it
%
% Software homepage: http://www.unife.it/prisma/software
%
% Copyright (C) 2011 by R. Cavicchioli, R. Zanella, G. Zanghirati, L. Zanni.
% ------------------------------------------------------------------------------
% COPYRIGHT NOTIFICATION
%
% Permission to copy and modify this software and its documentation for 
% internal research use is granted, provided that this notice is retained 
% thereon and on all copies or modifications. The authors and their
% respective Universities makes no representations as to the suitability 
% and operability of this software for any purpose. It is provided "as is"
% without express or implied warranty. Use of this software for commercial
% purposes is expressly prohibited without contacting the authors.
%
% This program is free software; you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation; either version 3 of the License, or (at your 
% option) any later version.
%
% This program is distributed in the hope that it will be useful, but 
% WITHOUT ANY WARRANTY; without even the implied warranty of 
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
% See the GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License along 
% with this program; if not, either visite http://www.gnu.org/licenses/
% or write to
% Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
% ==============================================================================

% start the clock
t0 = cputime;

% test for number of required parametres
if (nargin-length(varargin)) ~= 2
    error('Wrong number of required parameters');
end

%%%%%%%%%%%%%%%%%%%%%%%%
% SGP default parameters
%%%%%%%%%%%%%%%%%%%%%%%%
MAXIT = 1000;                       % maximum number of iterations
gamma = 1e-4;                   	% for sufficient decrease
beta = 0.4;                     	% backtracking parameter
M = 1;                          	% memory in obj. function value (if M = 1 monotone)
alpha_min = 1e-5;             		% alpha lower bound
alpha_max = 1e5;					% alpha upper bound
Malpha = 3;                     	% alfaBB1 memory
tau = 0.5;                      	% alternating parameter
initalpha = 1.3;                  	% initial alpha
initflag = 0;                       % 0 -> initial x all zeros
errflag = false;                    % 0 -> no error calculation
err = [];
verb = 0;                           % 0 -> silent
stopcrit = 1;                       % 1 -> number of iterations
bg = 0;                             % background value
omega = 'nonneg';                   % non negativity constraints
AT = [];                            % function handle for A'*x
dest_dir = '';
mysave = false;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Read the optional parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if (rem(length(varargin),2)==1)
    error('Optional parameters should always go by pairs');
else
    for i=1:2:(length(varargin)-1)
        switch upper(varargin{i})
            case 'AT'
                AT = varargin{i+1};
            case 'OBJ'
                obj = varargin{i+1};
                errflag = true;
            case 'INITIALIZATION'
                if numel(varargin{i+1}) > 1   % initial x provided by user
                    initflag = 999;
                    x = varargin{i+1};
                else
                    initflag = varargin{i+1};
                end
            case 'MAXIT'
                MAXIT = varargin{i+1};
            case 'M'
                M = varargin{i+1};
            case 'GAMMA'
                gamma = varargin{i+1};
            case 'BETA'
                beta = varargin{i+1};
            case 'ALPHA_MIN'
                alpha_min = varargin{i+1};
            case 'ALPHA_MAX'
                alpha_max = varargin{i+1};
            case 'MALPHA'
                Malpha = varargin{i+1};
            case 'TAUALPHA'
                tau = varargin{i+1};
            case 'INITALPHA'
                initalpha = varargin{i+1};
            case 'VERBOSE'
                verb = varargin{i+1};
            case 'TOL'
                tol = varargin{i+1};
            case 'STOPCRITERION'
                stopcrit = varargin{i+1};
            case 'BG'
                bg = varargin{i+1};
            case 'OMEGA'
                omega = varargin{i+1};
            case 'SAVE'
                mysave = true;
                dest_dir = varargin{i+1};    
            otherwise
                error(['Unrecognized option: ''' varargin{i} '''']);
        end;
    end;
end

%%%%%%%%%%%%%%%%%%
% function handles
%%%%%%%%%%%%%%%%%%
if ( isa(A,'function_handle') )
    if ( isempty(AT) )
        error('Missing parameter: AT');
    end
    if( ~isa(AT,'function_handle') )
        error('AT is not a function handle');
    end
else
    % Check column-normalization condition on A
    sumColsA = sum(A)';
    tolCheckA = 1.0e4*eps;
    checkA = find(abs(sumColsA-1) > tolCheckA); 
    if (~isempty(checkA))
        errmsg = sprintf('\n\t%d %s\n\t%s %d:\n\t%s%d%s%e,  %s%e',...
                         length(checkA),...
                         'not-normalized columns found in blurring matrix A.',...
                         'The first one is column',checkA(1),...
                         '|sum(A(:,',checkA(1),')) - 1| = ',...
                         abs(sumColsA(checkA(1))-1),'tolerance = ',tolCheckA);
        error('Not-normalized blurring matrix A: %s%s',...
              'provide a normalized A (see documentation).',errmsg);
    end
    
    AT = @(x) A'*x;
    A = @(x) A*x;
end

%%%%%%%%%%%%%%%%
% starting point
%%%%%%%%%%%%%%%%
switch initflag
    case 0          % all zeros
        x = zeros(size(gn));
    case 1          % random
        x = randn(size(gn));
    case 2          % gn
        x = gn;
    case 3          % same flux as gn - bg
        x = sum(gn(:) - bg)/numel(gn)*ones(size(gn));
    case 999        % x is explicitly given, check dimension
        if  not( size(x) == size(gn) )
            error('Invalid size of the initial point.');
        end
    otherwise
        error('Unknown initialization option.');
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% every image is treated as a vector
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
obj_size = size(x); 
x = x(:);
gn = gn(:);

%%%%%%%%%%%%%%%%
% stop criterion
%%%%%%%%%%%%%%%%
if not( ismember(stopcrit, [1 2 3 4]) )
    error('Unknown stopping criterion: ''%d''',num2str(stopcrit));
end
switch stopcrit
    case 1
        tol=[];
    case { 2 , 3 }
        if not(exist('tol','var'))
            tol=1e-4;
        end
    case 4
        if not(exist('tol','var'))
            tol=1+1/mean(gn);
        end
end

if (verb > 0)
   par_print();
end

%%%%%%%%%%%%%%
% data scaling
%%%%%%%%%%%%%%
scaling = max(gn(:));
gn = gn/scaling;
bg = bg/scaling;
x = x/scaling;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% change the null pixels of the observed image
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 
vmin = min(gn(gn>0));
gn(gn<=0) = vmin*eps*eps;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% some computations needed only once
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
N = numel(gn);                          % pixels in the image
flux = sum(gn) - N * bg;                % exact flux
iter = 1;                               % iteration counter
alpha = initalpha;                      % initial alpha
Valpha = alpha_max * ones(Malpha,1);    % memory buffer for alpha
Fold = -1e30 * ones(M, 1);              % memory buffer for obj. func.
Discr_coeff = 2/N*scaling;              % discrepancy coefficient
ONE = ones(N,1);

%%%%%%%%%%%%%%%%%
% projection type
%%%%%%%%%%%%%%%%%
switch (omega)
    case 'nonneg'
        pflag = 0;
    case 'nonneg_eq'
        pflag = 1;
    otherwise
        error('projection %s is not implemented',omega);
end

%%%%%%%%%%%%%%%%%%%%%
% output dir handling
%%%%%%%%%%%%%%%%%%%%%
if mysave
   [success, mess] = mkdir(dest_dir);
   if not(success)
       error('%s: %s',dest_dir,mess);
   end
   if not(isempty(mess))
       fprintf('%s\n\n',mess);
   end
end

%%%%%%%%%%%%%%%%%%%
% vector allocation
%%%%%%%%%%%%%%%%%%%
if errflag
    err = zeros(MAXIT+1,1);
    obj = obj(:);
    obj = obj/scaling;    
    obj_sum = sum(obj.*obj);
end
discr = zeros(MAXIT+1,1);
times = zeros(MAXIT+1,1);
times(1) = 0;

%%%%%%%%%%%%%%
% start of SGP
%%%%%%%%%%%%%%
% projection of the initial point
switch (pflag)
    case 0 % non negativity
        x( x < 0 ) = 0;        
    case 1 % non negativity and flux conservation
		% we have no diagonal scaling matrix yet, so 
		% we project using euclidean norm
		[x, biter, siter, r] = projectDF(flux, x, ones(size(x)));
end

% error
if errflag
    e = x - obj;
    err(1) = sqrt(sum(e.*e)/obj_sum);
end

% objective function value
x_tf = A(x);
den = x_tf + bg;
temp = gn./den;
g = ONE - AT(temp);
fv = sum( gn.* log(temp) ) + sum( x_tf ) - flux;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% bounds for the scaling matrices
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
y = (flux/(flux+N*bg)).*AT(gn);
X_low_bound = min(y(y>0));         	% Lower bound for the scaling matrix
X_upp_bound = max(y);               % Upper bound for the scaling matrix
if X_upp_bound/X_low_bound < 50
     X_low_bound = X_low_bound/10;
     X_upp_bound = X_upp_bound*10;
end

% discrepancy
discr(1) = Discr_coeff * fv;

% scaling matrix
if initflag == 0
    X = ones(size(x));
else
    X = x;
    % bounds
    X( X < X_low_bound ) = X_low_bound;
    X( X > X_upp_bound ) = X_upp_bound;
end

if pflag == 1
    D = 1./X;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% tolerance for stop criterion
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
switch stopcrit
   case 2
        if verb > 1
            fprintf('it %03d || x_k - x_(k-1) ||^2 / || x_k ||^2 %e \n',iter-1,0);
        end
        % instead of using sqrt each iteration
        tol = tol*tol;
   case 3
        if verb > 1
            fprintf('it %03d | f_k - f_(k-1) | / | f_k | %e \n',iter-1,0);
        end
   case 4
        if verb > 1
            fprintf('it %03d D_k %e \n',iter-1,discr(1));
        end
end

%%%%%%%%%%%
% main loop
%%%%%%%%%%%
loop = true;
while loop
    % store alpha and objective function values
    Valpha(1:Malpha-1) = Valpha(2:Malpha);
    Fold(1:M-1) = Fold(2:M);
    Fold(M) = fv;

    % compute descent direction 
    y = x - alpha*X.*g;
    
    switch (pflag) % projection onto the feasible set
        case 0 % non negativity
            y( y < 0 ) = 0;            
        case 1 % non negativity and flux conservation
            [y, biter, siter, r] = projectDF(flux, y.*D, D);
    end

    d = y - x;

    % backtracking loop for linesearch
    gd = dot(d,g);
    lam = 1;
    
    fcontinue = 1;
    d_tf = A(d);    % exploiting linearity
    fr = max(Fold);

    while fcontinue
        xplus = x + lam*d;
        
        x_tf_try = x_tf + lam*d_tf;
        den = x_tf_try + bg;
        
        temp = gn./den;
        fv = sum( gn.* log(temp) ) + sum( x_tf_try ) - flux;
        
        if ( fv <= fr + gamma * lam * gd || lam < 1e-12)
            x = xplus; clear xplus;
            sk = lam*d;
            x_tf = x_tf_try; clear x_tf_try;
            gtemp = ONE - AT(temp);

            yk = gtemp - g;
            g = gtemp; clear gtemp;
            fcontinue = 0;
        else
            lam = lam * beta;
        end
    end
    if (fv >= fr) && (verb > 0)
        disp('Worning: fv >= fr');
    end

    % update the scaling matrix and the steplength 
    X = x;
    X( X < X_low_bound ) = X_low_bound;
    X( X > X_upp_bound ) = X_upp_bound;

    D = 1./X;
    sk2 = sk.*D; yk2 = yk.*X;

    bk = dot(sk2,yk);  ck = dot(yk2,sk);
    if (bk <= 0)
        alpha1 = min(10*alpha,alpha_max);
    else
        alpha1BB = sum(dot(sk2,sk2))/bk;
        alpha1 = min(alpha_max, max(alpha_min, alpha1BB));
    end
    if (ck <= 0)
        alpha2 = min(10*alpha,alpha_max);
    else
        alpha2BB = ck/sum(dot(yk2,yk2));
        alpha2 = min(alpha_max, max(alpha_min, alpha2BB));
    end

    Valpha(Malpha) = alpha2;
    
    if (iter <= 20)
        alpha = min(Valpha);
    elseif (alpha2/alpha1 < tau)
        alpha = min(Valpha);
        tau = tau*0.9;
    else
        alpha = alpha1;
        tau = tau*1.1;
    end
    
    alpha = double(single(alpha));

    iter = iter + 1;
    times(iter) = cputime - t0;

    if errflag
        e = x - obj;
        err(iter) = sqrt(sum(e.*e)/obj_sum);
    end

    discr(iter) = Discr_coeff * fv;

    %%%%%%%%%%%%%%%
    % stop criteria
    %%%%%%%%%%%%%%%
    switch stopcrit
        case 1
            if verb > 1
                fprintf('it %03d of  %03d\n',iter-1,MAXIT);
            end
        case 2
            normstep = dot(sk,sk) / dot(x,x);
            loop = (normstep > tol);
            if verb > 1
                fprintf('it %03d || x_k - x_(k-1) ||^2 / || x_k ||^2 %e tol %e\n',iter-1,normstep,tol);
            end
        case 3
            reldecrease = abs(fv - Fold(M)) / abs(fv);
            loop = (reldecrease > tol);
            if verb > 1
                fprintf('it %03d | f_k - f_(k-1) | / | f_k | %e tol %e\n',iter-1,reldecrease,tol);
            end
        case 4
            loop = (discr(iter) > tol);
            if verb > 1
                fprintf('it %03d D_k %e tol %e\n',iter-1,discr(iter),tol);
            end
    end

    if iter > MAXIT
        loop = false;
    end
    
    %%%%%%%%
    % images
    %%%%%%%%
    if mysave
        % reconstructed image
        filename=sprintf('%s%crec_%03d.fits',dest_dir,filesep,iter-1);
        fitswrite((reshape(x,obj_size))',filename);
        % residual image
        res = (x - gn)./sqrt(x);
        filename=sprintf('%s%cPuetter_res_%03d.fits',dest_dir,filesep,iter-1);
        fitswrite((reshape(res,obj_size))',filename);
    end
    
end
x = reshape(x,obj_size);
x = x * scaling;

if errflag
    err = err(1:iter);
end
discr = discr(1:iter);
times = times(1:iter);
iter = iter - 1;

return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function par_print()
% par_print - Utility inner function to print-out parameters' values

initflag = evalin('caller','initflag');
switch initflag
    case 0
        temp = 'zeros';
    case 1
        temp = 'randn';
    case 2
        temp = 'gn';
    case 3
        temp = 'const';
    case 999
        temp = 'input';
    otherwise
        error(['Unrecognized initflag ''' num2str(initflag) '''']);
end
fprintf('Initialization %s\n',temp);
fprintf('MAXIT          %d\n',evalin('caller','MAXIT'));
M = evalin('caller','M');
if M > 1
    temp = 'nonmonotone';
else
    temp = 'monotone';
end
fprintf('M              %d (%s)\n',M,temp);
fprintf('gamma          %e\n',evalin('caller','gamma'));
fprintf('beta           %f\n',evalin('caller','beta'));
fprintf('alpha_min      %e\n',evalin('caller','alpha_min'));
fprintf('alpha_max      %e\n',evalin('caller','alpha_max'));
fprintf('Malpha         %d\n',evalin('caller','Malpha'));
fprintf('tau            %f\n',evalin('caller','tau'));
fprintf('initalpha      %f\n',evalin('caller','initalpha'));
fprintf('background     %f\n',evalin('caller','bg'));

errflag = evalin('caller','errflag');
switch errflag
    case 0
        temp = 'false';
    case 1
        temp = 'true';
    otherwise
        error(['unrecognized errflag ''' num2str(errflag) '''']);
end
fprintf('err calc       %s\n',temp);
fprintf('verbosity      %d\n',evalin('caller','verb'));
stopcrit = evalin('caller','stopcrit');
switch stopcrit
    case 1
        temp = 'iterations';
    case 2
        temp = 'relstepx or iterations';
    case 3
        temp = 'relstepf or iterations';
    case 4
        temp = 'discr. val. or iterations';
end
fprintf('stop criterion %s\n',temp);
fprintf('tolerance      %e\n',evalin('caller','tol'));
fprintf('constraints    %s\n',evalin('caller','omega'))
mysave = evalin('caller','mysave');
if mysave
    dest_dir = sprintf('%s%s%s',pwd,filesep,evalin('caller','dest_dir'));
else
    dest_dir = 'no';
end
fprintf('save           %s\n',dest_dir);

return
% ==============================================================================
% End of sgp_deblurring.m file - IRMA package
% ==============================================================================

