; Demo script to test the SGP method on deblurring problems
; Data from .fits files
; ------------------------------------------------------------------------------
;
; This software is developed within the research project
;
;        PRISMA - Optimization methods and software for inverse problems
;                           http://www.unife.it/prisma
;
; funded by the Italian Ministry for University and Research (MIUR), under
; the PRIN2008 initiative, grant n. 2008T5KA4L, 2010-2012. This software is
; part of the package "IRMA - Image Reconstruction in Microscopy and Astronomy"
; currently under development within the PRISMA project.
;
; Version: 1.0
; Date:    July 2011

; Authors:
;   Roberto Cavicchioli, Marco Prato, Luca Zanni
;    Dept. of Pure Appl. Math., Univ. of Modena and Reggio Emilia, Italy
;    roberto.cavicchioli@unimore.it, marco.prato@unimore.it, luca.zanni@unimore.it
;   Mario Bertero, Patrizia Boccacci
;    DISI (Dipartimento di Informatica e Scienze dell'Informazione), University of Genova, Italy
;    bertero@disi.unige.it, boccacci@disi.unige.it
;
; Software homepage: http://www.unife.it/prisma/software
;
; Copyright (C) 2011 by M. Bertero, P. Boccacci, R. Cavicchioli, M. Prato, L. Zanni.
; ------------------------------------------------------------------------------
; COPYRIGHT NOTIFICATION
;
; Permission to copy and modify this software and its documentation for
; internal research use is granted, provided that this notice is retained
; thereon and on all copies or modifications. The authors and their
; respective Universities makes no representations as to the suitability
; and operability of this software for any purpose. It is provided "as is"
; without express or implied warranty. Use of this software for commercial
; purposes is expressly prohibited without contacting the authors.
;
; This program is free software; you can redistribute it and/or modify it
; under the terms of the GNU General Public License as published by the
; Free Software Foundation; either version 3 of the License, or (at your
; option) any later version.
;
; This program is distributed in the hope that it will be useful, but
; WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
; See the GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License along
; with this program; if not, either visite http://www.gnu.org/licenses/
; or write to
; Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
; ==============================================================================

pro test_run

close, /all

filesep = path_sep()
;work_dir='C:'+filesep+'Documents and Settings'+filesep+'utente'+filesep+'IDLWorkspace'+filesep+'SGP'+filesep
work_dir='C:\Users\Marco\Desktop\IDL\'

inpath = work_dir+'input'+filesep
outpath = work_dir+'output'+filesep
path= 'NGC6946_single'+filesep+'Q1'+filesep ; Path to the folder that contains input files
;path= 'NGC7027_multi'+filesep
;path = 'star_cluster'+filesep
gn_name     = 'gn.fits'          ; file name of observed data
psf_name    = 'psf.fits'         ; file name of PSF image
bg_name     = 'bg.fits'          ; file name of background value
obj_name    = 'obj.fits'         ; file name of real data (optional)
maxiter     = 10                 ; maximum number of iterations
bound_flag  = 1B                 ; flag for bound correction
verbosity   = 1B                 ; verbosity level
algo_name   = 'S'                ; algorithm to be used ('S'=SGP, 'R'=RL, 'O'=OSEM)
GPU         = 0B                 ; GPU acceleration flag
tol         = 1e-7               ; tolerance value for stop criterion neq 1
stopcrit    = 1                  ; stop criterion
SGP_init    = 2					 ; SGP initialization
magnflag    = 0B                 ; magnitude calculation
;;; Magnitude parameters - binary
;stars_coord = [[123,132],[125,134],[129,126],[131,128]]  ; stars coordinates
;delta_t_m = 600   ; integration time
;magn_true = [20.,10.]   ; true magnitudes (if available)
;;; Magnitude parameters - star cluster
;stars_coord = [[154,155,227,276,339,342,325,298,295],[156,157,229,278,341,344,327,300,297],$  ; stars
;               [252,236,245,215,246,279,312,294,329],[254,238,247,217,248,281,314,296,331]]   ; coordinates
;delta_t_m = 120  ; integration time
;magn_true = [15.09,13.62,12.86,14.17,13.70,15.44,14.29,13.86,15.64]   ; true magnitudes (if available)

if GPU then begin
   out_hard = 'GPU'
endif else begin
   out_hard = 'CPU'
endelse

if algo_name eq 'R' then begin
   out_lab = 'RL'
endif else begin
   if algo_name eq 'O' then begin
      out_lab = 'OSEM'
   endif else begin
      if algo_name eq 'S' then begin
         out_lab = 'SGP'
      endif
   endelse
endelse

fits_read, inpath + path + gn_name, gn     ; noisy image
fits_read, inpath + path + psf_name, psf   ; PSF
fits_read, inpath + path + bg_name, bg_or  ; background
bg = bg_or[0]
bg = double(bg)
gn = double(gn)
psf = double(psf)
err_calc = 0B
if N_ELEMENTS(obj_name) ne 0 then begin
   fits_read, inpath + path + obj_name, obj_input
   err_calc = 1B
   obj_input = double(obj_input)
endif

dims = n_elements(size(gn,/dimensions))
if (dims eq 2) then begin
   test_deblurring_single, psf, gn, x, iter, err, discr, times, maxit = maxiter, $
                           bound = bound_flag, algo = algo_name, verb=verbosity, $
                           stopcrit = stopcrit, tol = tol, obj = obj_input, $
                           GPU = GPU, bg = bg, init_case = SGP_init
endif else begin
   test_deblurring_multi, psf, gn, x, iter, err, discr, times, maxit = maxiter, $
                          bound = bound_flag, algo = algo_name, verb=verbosity, $
                          stopcrit = stopcrit, tol = tol, obj = obj_input, $
                          GPU = GPU, bg = bg, init_case = SGP_init
endelse

; time tracking
print, 'Elapsed time: ', times[n_elements(times)-1]

dim = size(x,/dimensions)
window, 1, xsize=dim[0], ysize=dim[1], title='Rec'
tvscl, rotate(x,3)

if err_calc then begin
   window, 11, title='Reconstruction error'
   plot, err
   m = min(err,ind_min)
   print, 'Min err: ', m, ' at iteration ', ind_min
   if stopcrit eq 1 then  print, 'Last iteration err=', err[maxiter]
   window, 2, xsize=dim[0], ysize=dim[1], title='Obj'
   tvscl, rotate(obj_input,3)
endif

if magnflag then begin
   dim_coord = size(stars_coord,/dimensions)
   N_stars = dim_coord[0]
   magn = dblarr(N_stars)
   for is = 0, N_stars-1 do begin
       magn[is] = magnitude(total(x[stars_coord[is,0]:stars_coord[is,1], $
                            stars_coord[is,2]:stars_coord[is,3]],/DOU)/0.3, $
                            band='K',surf=104,delta_t=delta_t_m)
   endfor
   if N_ELEMENTS(magn_true) ne 0 then begin
      av_rel_err = mean(abs(magn-magn_true)/magn_true)
      print, 'Average relative error = ', av_rel_err
   endif
endif else begin
   magn = 0
endelse

if bound_flag then begin
   save, filename=outpath+path+'out_'+out_lab+'_'+out_hard+'_it'+string(maxiter,format='(I05)')+'_bound.sav', $
         x, iter, err, magn, discr, times
endif else begin
   save, filename=outpath+path+'out_'+out_lab+'_'+out_hard+'_it'+string(maxiter,format='(I05)')+'.sav', $
         x, iter, err, magn, discr, times
endelse

END

; ==============================================================================
; End of test_run.pro file - IRMA package
; ==============================================================================