PRO sgp_deblurring, A, gn, x, iter, err, Discr, times, bg = bg, bound = bound, $
					initialization = initialization, maxit = maxit, obj = obj, $
					stopcrit = stopcrit, tol = tol, verb = verb

; sgp_deblurring - SGP algorithm for non-regularized deblurring
;   This function solves an image deblurring problem by applying the SGP
;   algorithm to the minimization of the generalized Kullback-Leibler
;   divergence with no regularization [1]:
;
;      min KL(A*x + bg, gn)
;       x in OMEGA
;
;   where KL(u,v) is the generalized Kullback-Leibler divergence between
;   vectors u and v, bg is the background, gn are the observed data and
;   the feasible set OMEGA is x(i) >= 0;
;
;   [1] S. Bonettini, R. Zanella, L. Zanni,
;       "A scaled gradient projection method for constrained image deblurring",
;       Inverse Problems 25(1), 2009, January, 015002.
;
; SYNOPSIS
;    sgp_deblurring, A, gn, x, iter, err, Discr, times [, opts]
;
; MANDATORY INPUT
;   A   (double array) - measuring matrix used to apply the blurring
;                        operator, that is to compute A*x
;                        REMARK: all comlumns of A must sum-up to 1
;   gn  (double array) - measured image
;
; OPTIONAL INPUT
;   The following options must be provided as keyword/value pairs.
;   'OBJ'              - Exact solution, for error calculation (double array)
;   'BG'               - Background value (double)
;                        DEFAULT = 0
;   'INITIALIZATION'   - Choice for starting point:
;                        0  - all zero starting point
;                        1  - initialization with gn
;                        2  - initialization with
;                               ones(size(gn))*sum(gn(:) - bg) / numel(gn)
;                        x0 - user-provided starting point (double array)
;                        DEFAULT = 2
;   'MAXIT'            - Maximum number of iterations (integer)
;                        DEFAULT = 1000
;   'VERB'             - Verbosity level (integer)
;                        0 - silent
;                        1 - print configuration parameters at startup and
;                            some information at each iteration
;                        DEFAULT = 0
;   'STOPCRIT'         - Choice for stopping rule (integer)
;                        1 -> iter > MAXIT
;                        2 -> ||x_k - x_(k-1)|| <= tol*||x_k|| OR iter > MAXIT
;                        3 -> |KL_k - KL_(k-1)| <= tol*|KL_k| OR iter > MAXIT
;                        4 -> (2/N)*KL_k <= tol OR iter > MAXIT
;                        DEFAULT = 1;
;   'TOL'              - Tolerance used in the stopping criterion
;                        DEFAULT = 1e-4 if STOPCRITERION = 2 or 3
;                        DEFAULT = 1+1/mean(gn) if STOPCRITERION = 4
;   'BOUND'            - Flag to enable bound effect reduction
;
; OUTPUT
;   x                  - Reconstructed data
;   iter               - Number of iterations
;   err                - Error value at each iteration. If OBJ was not given,
;                        then err is the empty matrix.
;   discr              - Discrepancy value after each iteration:
;                            D = 2/numel(x_k) * KL( Ax_k + bg, gn)
;   times              - Time elapsed after each iteration
;
; ------------------------------------------------------------------------------
;
; This software is developed within the research project
;
;        PRISMA - Optimization methods and software for inverse problems
;                           http://www.unife.it/prisma
;
; funded by the Italian Ministry for University and Research (MIUR), under
; the PRIN2008 initiative, grant n. 2008T5KA4L, 2010-2012. This software is
; part of the package "IRMA - Image Reconstruction in Microscopy and Astronomy"
; currently under development within the PRISMA project.
;
; Version: 1.0
; Date:    July 2011

; Authors:
;   Roberto Cavicchioli, Marco Prato, Luca Zanni
;    Dept. of Pure Appl. Math., Univ. of Modena and Reggio Emilia, Italy
;    roberto.cavicchioli@unimore.it, marco.prato@unimore.it, luca.zanni@unimore.it
;   Mario Bertero, Patrizia Boccacci
;    DISI (Dipartimento di Informatica e Scienze dell'Informazione), University of Genova, Italy
;    bertero@disi.unige.it, boccacci@disi.unige.it
;
; Software homepage: http://www.unife.it/prisma/software
;
; Copyright (C) 2011 by M. Bertero, P. Boccacci, R. Cavicchioli, M. Prato, L. Zanni
; ------------------------------------------------------------------------------
; COPYRIGHT NOTIFICATION
;
; Permission to copy and modify this software and its documentation for
; internal research use is granted, provided that this notice is retained
; thereon and on all copies or modifications. The authors and their
; respective Universities makes no representations as to the suitability
; and operability of this software for any purpose. It is provided "as is"
; without express or implied warranty. Use of this software for commercial
; purposes is expressly prohibited without contacting the authors.
;
; This program is free software; you can redistribute it and/or modify it
; under the terms of the GNU General Public License as published by the
; Free Software Foundation; either version 3 of the License, or (at your
; option) any later version.
;
; This program is distributed in the hope that it will be useful, but
; WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
; See the GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License along
; with this program; if not, either visite http://www.gnu.org/licenses/
; or write to
; Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
; ==============================================================================

; start the clock
t0 = systime(2)

;;;;;;;;;;;;;;;;;;;;;;;;;;
; SGP default parameters ;
;;;;;;;;;;;;;;;;;;;;;;;;;;
alpha_min = 1.e-5        ; alpha lower bound
alpha_max = 1.e5		     ; alpha upper bound
theta = 0.4              ; backtracking parameter
beta = 1.e-4             ; for sufficient decrease
initalpha = 1.3          ; initial alpha
M = 1                    ; memory in obj. function value (if M = 1 monotone)
Malpha = 3               ; alfaBB1 memory
tau = 0.5                ; alternating parameter
initflag = 2             ; 2 -> constant image
errflag = 0B             ; 0 -> no error calculation
err = -1

if (keyword_set(bg) - 1)          then bg = 0.                  ; background value
if (keyword_set(bound) - 1)       then bound = 0B               ; bound effects
if (keyword_set(maxit) - 1)       then maxit = 1000             ; maximum number of iterations
if (keyword_set(stopcrit) - 1)    then stopcrit = 1             ; 1 -> number of iterations
if (keyword_set(verb) - 1)        then verb = 0                 ; 0 -> silent

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
dim = size(gn,/dimensions)
if (size(gn,/n_dimensions) eq 2) then dim = [dim,1]

if bound then begin
   newdim = [2^(floor(alog(dim[0])/alog(2))+1) ,  2^(floor(alog(dim[1])/alog(2))+1)]
   margine = (newdim - dim[0:1])/2
   ; WARNING: at this level, if A and gn differ in dimensions, then
   ; the input A MUST be sized as newdim[0] x newdim[1] by the user
   dim_A = size(A,/dimensions)
   if n_elements(dim_A) eq 2 then dim_A = [dim_A,1]
   if array_equal(dim_A,dim) then begin
      psf_work = dblarr(newdim[0],newdim[1],dim[2])
      psf_work[margine[0]:margine[0]+dim[0]-1,margine[1]:margine[1]+dim[1]-1,*] = A
   endif else begin
      psf_work = A
   end
   TF = dcomplexarr(newdim[0],newdim[1],dim[2])
   CTF = TF
      for i = 0, dim[2]-1 do begin
          TF[*,*,i] = fft(shift(reform(psf_work[*,*,i]),newdim/2))*newdim[0]*newdim[1]
          CTF[*,*,i] =conj(TF[*,*,i])
      endfor
endif else begin
   TF = dcomplexarr(dim)
   CTF = TF
      for i = 0, dim[2]-1 do begin
          TF[*,*,i] = fft(shift(reform(A[*,*,i]),dim[0:1]/2))*dim[0]*dim[1]
          CTF[*,*,i] = conj(TF[*,*,i])
      endfor
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Read the optional parameters ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
if arg_present(obj) then errflag = 1B
if keyword_set(initialization) then begin
	if product(size(initialization,/dimensions)) gt 1 then begin
		initflag = 999
		x = initialization		; initial x provided by user
	endif else begin
		initflag = initialization
	endelse
endif

;;;;;;;;;;;;;;;;;;
; starting point ;
;;;;;;;;;;;;;;;;;;

case initflag of
    0: x = dblarr(dim[0],dim[1])					; all zeros
    1: x = gn[*,*,0]											; gn
    2: x = total(gn-bg)/n_elements(gn)*make_array(dim[0],dim[1],value=1.,/double)	; same flux as gn - bg
    999: if not(array_equal(size(x,/dimensions),dim[0:1])) then	begin   ; x is explicitly given, check dimensions
            message, 'Invalid size of the initial point.'
         endif
    else: message, 'Unknown initialization option.'
endcase

; size of the images
obj_size = dim[0:1]
if bound then begin
	work_size = newdim
endif else begin
	work_size = obj_size
endelse

;;;;;;;;;;;;;;;;;;
; stop criterion ;
;;;;;;;;;;;;;;;;;;

if (stopcrit ne 1 and stopcrit ne 2 and stopcrit ne 3 and stopcrit ne 4) then begin
    message, 'Unknown stopping criterion:', stopcrit
end

if (keyword_set(tol) - 1) then begin
	if (stopcrit eq 2 or stopcrit eq 3) then tol = 1.e-4
	if (stopcrit eq 4) then tol = 1. + 1./mean(gn)
end

;;;;;;;;;;;;;;;;
; data scaling ;
;;;;;;;;;;;;;;;;
scaling = max(gn)
gn = gn/scaling
bg = bg/scaling
x = x/scaling

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; change the null pixels of the observed image ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
par = machar(/double)
vmin = min(gn(where(gn gt 0)))
gn_nonpos = where(gn le 0, n_nonpos)
if (n_nonpos gt 0) then gn(gn_nonpos) = vmin*par.eps^2

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; initializations and computations that need only once ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
N = n_elements(gn)/dim[2]                 		; pixels in the image
flux = total(total(gn,1),1) - N * bg      		; exact flux
iter = 1                                		  ; iteration counter
alpha = initalpha                       		  ; initial alpha
Valpha = alpha_max * (dblarr(Malpha)+1.) 		  ; memory buffer for alpha
Fold = -1.e30 * (dblarr(M)+1.)             		  ; memory buffer for obj. func.
Discr_coeff = 2./(N*dim[2])*scaling               ; discrepancy coefficient
ONE = make_array(work_size,value = dim[2], /double)

if bound then begin
	Mask = dblarr(work_size)
	Mask[margine[0]:margine[0]+dim[0]-1,margine[1]:margine[1]+dim[1]-1] = 1.
	work_index = long(where(Mask gt 0.))
	sigma = 1.e-2
	  Weight = afunction(Mask,CTF)
  no_z = where(Weight[*,*,0] gt sigma)
  if dim[2] gt 1 then begin
    tmp = intarr(newdim[0],newdim[1])
    tmp(no_z)=1
  for i = 1, dim[2]-1 do begin
    no_z_tmp = where(Weight[*,*,i] gt sigma)
    tmp2 = intarr(newdim[0],newdim[1])
    tmp2(no_z_tmp)=1
    tmp = tmp and tmp2
  endfor
    no_z = where(tmp eq 1)
    Weight = total(Weight,3)
  endif
    ATM = dblarr(work_size)
    ATM[no_z] = Weight[no_z]

    ;;; initial value of x
    xtmp = mean(flux)*dim[2]/total(Weight(no_z))
    x = dblarr(work_size)
    x[no_z] = xtmp

    Wtmp = dblarr(work_size)
    Wtmp(no_z) = dim[2]/Weight(no_z)
    Weight = Wtmp
endif else begin
  work_index = transpose(lindgen(product(work_size)))
  no_z = work_index
endelse

;;;;;;;;;;;;;;;;;;;;;
; vector allocation ;
;;;;;;;;;;;;;;;;;;;;;
if errflag then begin
   err = dblarr(maxit+1)
   obj = obj/scaling
   obj_sum = total(obj*obj)
endif
Discr = dblarr(maxit+1)
times = dblarr(maxit+1)

;;;;;;;;;;;;;;;;
; start of SGP ;
;;;;;;;;;;;;;;;;
; projection of the initial point
	 x_nonpos =  where(x lt 0, n_nonpos)
	 if (n_nonpos gt 0) then x(x_nonpos) = 0.

; error
if errflag then begin
   e = x(work_index) - obj[*]
   err[0] = sqrt(total(e*e)/obj_sum)
endif

if bound then begin
	 tmp = dblarr([work_size,dim[2]])
	 for i = 0, dim[2]-1 do begin
		   tmp[*,*,i] = Mask
	 endfor
	 multi_index = where(tmp gt 0)
	 tmp = dblarr(work_size[0],work_size[1],dim[2])
	 tmp[multi_index] = gn[*]
	 gn = tmp
endif else begin
	 multi_index = lindgen(dim[2]*product(work_size))
endelse

; objective function value
x_tf = afunction(x,TF)
den = x_tf + bg
temp = gn/den
g = afunction(temp,CTF)

if dim[2] gt 1 then begin
	g = total(g,3)
endif
if bound then begin
	g = ATM - g
endif else begin
	g = ONE - g
endelse
fv = total(gn(multi_index)*alog(temp(multi_index))) + total(x_tf(multi_index)) - total(flux)

;;;;;  bounds for the scaling matrices ;;;;;;;;
temp = afunction(gn,CTF)
tmp = temp[*,*,0]
y = (flux[0]/(flux[0] + N*bg))*tmp(work_index)
y_min = min(y[where(y gt 0)])
y_max = max(y)
for i = 1, dim[2]-1 do begin
	tmp = temp[*,*,i]
	y = (flux[i]/(flux[i] + N*bg))*tmp(work_index)
	y_min = min([min(y[where(y gt 0)]),y_min])
	y_max = max([max(y),y_max])
endfor

X_low_bound = y_min		         ; Lower bound for the scaling matrix
X_upp_bound = y_max    			   ; Upper bound for the scaling matrix

if X_upp_bound/X_low_bound lt 50. then begin
   X_low_bound = X_low_bound/10.
   X_upp_bound = X_upp_bound*10.
endif
; discrepancy
Discr[0] = Discr_coeff * fv

; scaling matrix
if initflag eq 0 then begin
	XX = make_array(size(x,/dimensions), value = 1., /double)
endif else begin
    XX = x
    ; bounds
    XX_toolow = where(XX lt X_low_bound, nlow)
    if (nlow gt 0) then XX(XX_toolow) = X_low_bound
    XX_toohigh = where(XX gt X_upp_bound, nhigh)
    if (nhigh gt 0) then XX(XX_toohigh) = X_upp_bound
    if bound then XX = Weight*XX
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; tolerance for stop criterion
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

loop = 1B
while loop do begin

    Valpha[0:Malpha-2] = Valpha[1:Malpha-1]
    if M gt 1 then Fold[0:M-2] = Fold[1:M-1]
    Fold[M-1] = fv

    ; Step 2.1
    y = x - alpha*XX*g

    ; projection
		y_nonpos = where(y lt 0, n_nonpos)
		if (n_nonpos gt 0) then y(y_nonpos) = 0

    d = y - x
    gd = total(d*g)
    lam = 1.

    ; Step 2.2
    fcontinue = 1B

    ; exploiting linearity
    d_tf = afunction(d,TF)

    fr = max(Fold)

    while fcontinue do begin
        xplus = x + lam*d

        x_tf_try = x_tf + lam*d_tf
        den = x_tf_try + bg

        temp = gn/den
        fv = total( gn(multi_index)* alog(temp(multi_index)) ) + total( x_tf_try(multi_index) ) - total(flux)

        ; Step 2.3
        if ( fv le fr + beta * lam * gd OR lam lt 1.e-12) then begin
            x = xplus	;clear xplus
            sk = lam*d
            x_tf = x_tf_try
            gtemp = afunction(temp,CTF)
            if dim[2] gt 1 then begin
            	gtemp = total(gtemp,3)
            endif
            if bound then begin
	            gtemp = ATM - gtemp
	        endif else begin
	        	gtemp = ONE - gtemp
	        endelse

            yk = gtemp - g
            g = gtemp
            fcontinue = 0B
        endif else begin
            lam = lam * theta
        endelse
    endwhile

    if (fv ge fr AND verb gt 0) then print, 'WARNING: fv >= fr'

    ; Step 3
    XX = x
    ; bounds
    XX_toolow = where(XX lt X_low_bound, nlow)
    if (nlow gt 0) then XX(XX_toolow) = X_low_bound
    XX_toohigh = where(XX gt X_upp_bound, nhigh)
    if (nhigh gt 0) then XX(XX_toohigh) = X_upp_bound
    if bound then XX = Weight*XX
    D = dblarr(work_size)
    D[no_z] = 1./XX[no_z]

    sk2 = sk*D
    yk2 = yk*XX

    bk = total(sk2*yk)
    ck = total(yk2*sk)

    if (bk le 0) then begin
        alpha1 = min([10.*alpha,alpha_max])
    endif else begin
        alpha1BB = total(sk2*sk2)/bk
        alpha1 = min([alpha_max, max([alpha_min, alpha1BB])])
    end
    if (ck le 0) then begin
        alpha2 = min([10.*alpha,alpha_max])
    endif else begin
        alpha2BB = ck/total(yk2*yk2)
        alpha2 = min([alpha_max, max([alpha_min, alpha2BB])])
    end

    Valpha[Malpha-1] = alpha2

    if (iter le 20) then begin
       alpha = min(Valpha)
    endif else begin
    	if (alpha2/alpha1 lt tau) then begin
	       alpha = min(Valpha)
       	   tau = tau*0.9
       	endif else begin
	        alpha = alpha1
    	    tau = tau*1.1
    	endelse
    endelse
    times[iter] = systime(2) - t0
    iter = iter + 1

    alpha = double(float(alpha))

    if errflag then begin
        e = x(work_index) - obj[*]
        err[iter-1] = sqrt(total(e*e)/obj_sum)
    endif

    Discr[iter-1] = Discr_coeff * fv

    ;;;;;;;;;;;;;;;;;
    ; stop criteria ;
    ;;;;;;;;;;;;;;;;;

    case stopcrit of
        1: begin
              if verb gt 0 then begin
                 print, 'it ', iter-1, ' of ', maxit
              endif
           end
        2: begin
        		  normstep = total(sk*sk)
           		loop = (normstep gt tol*total(x*x))
           		if verb gt 0 then begin
              	 print, 'it ', iter-1, ', || x_k - x_k-1 || ^2 / || x_k || ^2 ', normstep, ', tol ', tol
           		endif
           end
        3: begin
        		  reldecrease = abs(fv - Fold[M-1])/abs(fv)
           		loop = (reldecrease gt tol)
           		if verb gt 0 then begin
              	 print, 'it ', iter-1, ', | f_k - f_k-1 | / | f_k | ', reldecrease, ', tol ', tol
           		endif
           end
        4: begin
            if iter ne 1 then begin
        		  loop = (Discr[iter-1] gt tol)
            endif else begin
              loop = 1
            endelse
           		if verb gt 0 then begin
              		print, 'it ', iter-1, ', D_k ', Discr[iter-1], ', tol ', tol
           		endif
           end
    endcase

    if iter gt maxit then loop = 0B

    if verb gt 0 then begin
      print, 'Iteration:', iter,  '  Fobj:', fv, '  Alpha:', alpha, '  Lambda:', lam
    endif

endwhile

x = x(work_index)
x = reform(x,obj_size)
x = x * scaling

if errflag then err = err[0:iter-1]

Discr = Discr[0:iter-1]
times = times[0:iter-1]
iter = iter - 1

END

; ==============================================================================
; End of SGP_deblurring.pro file - IRMA package
; ==============================================================================