PRO rl_deblurring, A, gn, x, iter, err, Discr, times, maxit = maxit, bg = bg, bound = bound, $
                   obj = obj, stars_coord = stars_coord, delta_t = delta_t, $
                   stopcrit=stopcrit, tol = tol, verb = verb

; rl_deblurring - RL algorithm
;   This function solves an image deblurring problem by applying the RL
;   algorithm.
;
; SYNOPSIS
;    rl_deblurring, A, gn, x, iter, err, discr, times [, opts]
;
; MANDATORY INPUT
;   A   (double array) - measuring matrix used to apply the blurring
;                        operator, that is to compute A*x
;   gn  (double array) - measured image
;
; OPTIONAL INPUT
;   The following options must be provided as keyword/value pairs.
;   'OBJ'              - Exact solution, for error calculation (double array)
;   'BG'               - Background value (double)
;                        DEFAULT = 0
;   'MAXIT'            - Maximum number of iterations (integer)
;                        DEFAULT = 1000
;   'VERB'             - Verbosity level (integer)
;                        0 - silent
;                        1 - print configuration parameters at startup and
;                            some information at each iteration
;                        DEFAULT = 0
;   'STOPCRIT'         - Choice for stopping rule (integer)
;                        1 -> iter > MAXIT
;                        3 -> |KL_k - KL_(k-1)| <= tol*|KL_k| OR iter > MAXIT
;                        4 -> (2/N)*KL_k <= tol OR iter > MAXIT
;                        DEFAULT = 1;
;   'TOL'              - Tolerance used in the stopping criterion
;                        DEFAULT = 1e-4 if STOPCRITERION = 3
;                        DEFAULT = 1+1/mean(gn) if STOPCRITERION = 4
;
; OUTPUT
;   x                  - Reconstructed data
;   iter               - Number of iterations
;   err                - Error value at each iteration. If OBJ was not given,
;                        then err is the empty matrix.
;   discr              - Discrepancy value after each iteration:
;                            D = 2/numel(x_k) * KL( Ax_k + bg, gn)
;   times              - Time elapsed after each iteration
;
; ------------------------------------------------------------------------------
;
; This software is developed within the research project
;
;        PRISMA - Optimization methods and software for inverse problems
;                           http://www.unife.it/prisma
;
; funded by the Italian Ministry for University and Research (MIUR), under
; the PRIN2008 initiative, grant n. 2008T5KA4L, 2010-2012. This software is
; part of the package "IRMA - Image Reconstruction in Microscopy and Astronomy"
; currently under development within the PRISMA project.
;
; Version: 1.0
; Date:    July 2011

; Authors:
;   Roberto Cavicchioli, Marco Prato, Luca Zanni
;    Dept. of Pure Appl. Math., Univ. of Modena and Reggio Emilia, Italy
;    roberto.cavicchioli@unimore.it, marco.prato@unimore.it, luca.zanni@unimore.it
;   Mario Bertero, Patrizia Boccacci
;    DISI (Dipartimento di Informatica e Scienze dell'Informazione), University of Genova, Italy
;    bertero@disi.unige.it, boccacci@disi.unige.it
;
; Software homepage: http://www.unife.it/prisma/software
;
; Copyright (C) 2011 by M. Bertero, P. Boccacci, R. Cavicchioli, M. Prato, L. Zanni.
; ------------------------------------------------------------------------------
; COPYRIGHT NOTIFICATION
;
; Permission to copy and modify this software and its documentation for
; internal research use is granted, provided that this notice is retained
; thereon and on all copies or modifications. The authors and their
; respective Universities makes no representations as to the suitability
; and operability of this software for any purpose. It is provided "as is"
; without express or implied warranty. Use of this software for commercial
; purposes is expressly prohibited without contacting the authors.
;
; This program is free software; you can redistribute it and/or modify it
; under the terms of the GNU General Public License as published by the
; Free Software Foundation; either version 3 of the License, or (at your
; option) any later version.
;
; This program is distributed in the hope that it will be useful, but
; WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
; See the GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License along
; with this program; if not, either visite http://www.gnu.org/licenses/
; or write to
; Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
; ==============================================================================

; start the clock
t0 = systime(2)

;;;;;;;;;;;;;;;;;;;;;;;;;
; RL default parameters ;
;;;;;;;;;;;;;;;;;;;;;;;;;
errflag = 1B
err = 0
if (keyword_set(bg) - 1)          then bg = 0.           ; background value
if (keyword_set(bound) - 1)       then bound = 0B        ; bound effects
if (keyword_set(maxit) - 1)       then maxit = 1000      ; maximum number of iterations
if (keyword_set(stopcrit) - 1)    then stopcrit = 1      ; 1 -> number of iterations
if (keyword_set(verb) - 1)        then verb = 0          ; 0 -> silent
if (keyword_set(obj) - 1)         then errflag = 0B      ; error calculation

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
dim = size(gn,/dimensions)
if (size(gn,/n_dimensions) eq 2) then dim = [dim,1]

if bound then begin
   newdim = [2^(floor(alog(dim[0])/alog(2))+1) ,  2^(floor(alog(dim[1])/alog(2))+1)]
   margine = (newdim - dim[0:1])/2
   ;; WARNING: at this level, if A and gn differ in dimensions, then
   ;; the input A MUST be sized as newdim[0] x newdim[1] by the user
   if array_equal(size(A[*,*,0],/dimensions),dim[0:1]) then begin
      psf = A
      psf_work = dblarr(newdim[0],newdim[1],dim[2])
      psf_work[margine[0]:margine[0]+dim[0]-1,margine[1]:margine[1]+dim[1]-1,*] = psf
   endif else begin
      psf_work = A
   end
   TF = dcomplexarr(newdim[0],newdim[1],dim[2])
   CTF = TF
      for i = 0, dim[2]-1 do begin
          TF[*,*,i] = fft(shift(reform(psf_work[*,*,i]),newdim/2))*newdim[0]*newdim[1]
          CTF[*,*,i] =conj(TF[*,*,i])
      endfor
endif else begin
   TF = dcomplexarr(dim)
   CTF = TF
      for i = 0, dim[2]-1 do begin
          TF[*,*,i] = fft(shift(reform(A[*,*,i]),dim[0:1]/2))*dim[0]*dim[1]
          CTF[*,*,i] = conj(TF[*,*,i])
      endfor
end

; size of the images
obj_size = dim[0:1]
if bound then begin
	work_size = newdim
endif else begin
	work_size = obj_size
endelse

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; initializations and computations that need only once ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
N = n_elements(gn)/dim[2]                 		; pixels in the image
flux = total(total(gn,1),1) - N * bg      		; exact flux

if bound then begin
	Mask = dblarr(work_size)
	Mask[margine[0]:margine[0]+dim[0]-1,margine[1]:margine[1]+dim[1]-1] = 1.
	work_index = long(where(Mask gt 0.))
	sigma = 1.e-2
  	Weight = afunction(Mask,CTF)
	no_z = where(Weight[*,*,0] gt sigma)
	if dim[2] gt 1 then begin
	  tmp = intarr(newdim[0],newdim[1])
	  tmp(no_z)=1
  for i = 1, dim[2]-1 do begin
    no_z_tmp = where(Weight[*,*,i] gt sigma)
    tmp2 = intarr(newdim[0],newdim[1])
    tmp2(no_z_tmp)=1
    tmp = tmp and tmp2
  endfor
    no_z = where(tmp eq 1)
		Weight = total(Weight,3)
	endif

	;;; initial value of x
	xtmp = mean(flux)*dim[2]/total(Weight(no_z))
	rec = make_array(work_size, value = xtmp, /double)

	Wtmp = dblarr(work_size)
	Wtmp(no_z) = 1./Weight(no_z)
	Weight = Wtmp
endif else begin
	work_index = transpose(lindgen(product(work_size)))
	rec = make_array(work_size, value = mean(flux)*dim[2]/product(work_size), /double)
endelse

;;;;;;;;;;;;;;;;;;;
; vector allocation
;;;;;;;;;;;;;;;;;;;
if errflag then begin
    err = dblarr(maxit+1)
    obj_sum = total(obj*obj)
endif
Discr = dblarr(maxit+1)
times = dblarr(maxit+1)

;;;;;;;;;;;;;;;;;;
; stop criterion ;
;;;;;;;;;;;;;;;;;;
if (stopcrit ne 1 and stopcrit ne 3 and stopcrit ne 4) then begin
    message, 'Unknown stopping criterion:', stopcrit
end

if (keyword_set(tol) - 1) then begin
  if (stopcrit eq 3) then tol = 1.e-4
  if (stopcrit eq 4) then tol = 1. + 1./mean(gn)
end

;;;;;;;;;;;;;;;
; start of RL ;
;;;;;;;;;;;;;;;
if bound then begin
	tmp = dblarr([work_size,dim[2]])
	for i = 0, dim[2]-1 do begin
		tmp[*,*,i] = Mask
	endfor
	multi_index = where(tmp gt 0)
	tmp = dblarr(work_size[0],work_size[1],dim[2])
	tmp[multi_index] = gn[*]
	gn = tmp
endif else begin
	multi_index = lindgen(dim[2]*product(work_size))
endelse
iter=0
loop=1

while loop do begin
	if errflag then begin
		err[iter] = sqrt(total((rec[work_index]-obj[*])^2.)/obj_sum)
	endif

	; objective function value
  	x_tf = afunction(rec,TF)
	den = x_tf + bg
	temp = gn/den
	fv = total(gn(multi_index)*alog(temp(multi_index))) + total(x_tf(multi_index)) - total(flux)

	; discrepancy
	Discr[iter] = 2.*fv/(N*dim[2])

    tmp = afunction(temp,CTF)
	if dim[2] gt 1 then tmp = total(tmp,3)
	if bound then begin
		rec = Weight*rec*tmp
	endif else begin
		rec = rec*tmp/dim[2]
	endelse

    times[iter] = systime(2) - t0
    iter = iter + 1

     case stopcrit of
        1: if verb gt 0 then begin
              print, 'it ', iter-1, ' of ', maxit
           endif
        3: begin
            if iter ne 1 then begin
              reldecrease = abs(fv - Discr[iter-2]*(N*dim[2])/2.)/abs(fv)
            endif else begin
              reldecrease = 1
            endelse
              loop = (reldecrease gt tol)
              if verb gt 0 then begin
                  print, 'it ', iter-1, ', | f_k - f_k-1 | / | f_k | ', reldecrease, ', tol ', tol
              endif
           end
        4: begin
            if iter ne 1 then begin
              loop = (Discr[iter-1] gt tol)
            endif else begin
              loop = 1
            endelse
              if verb gt 0 then begin
                  print, 'it ', iter-1, ', D_k ', Discr[iter-1], ', tol ', tol
              endif
           end
    endcase

    if iter gt maxit then loop = 0B

endwhile

x = rec(work_index)
x = reform(x,obj_size)

END

; ==============================================================================
; End of RL_deblurring.pro file - IRMA package
; ==============================================================================
