function magnitude, flux, surf=surf, delta_t=delta_t,band=band

; magnitude - Calculates magnitude for a single star
;   This function calculate the magnitude of a given area of the image
;
; SYNOPSIS
;   out = magnitude(flux [, opts])
;
; MANDATORY INPUT
;   flux       (double)  - input total flux of the source
;
; OPTIONAL INPUT
;   The following options must be provided as keyword/value pairs.
;   'SURF'               - integrating surface
;   'DELTA_T'            - observation time amplitude
;   'BAND'               - electromagnetic band of the observation
;
; OUTPUT
;   out        (double)        - calculated magnitude
;
; ------------------------------------------------------------------------------
;
; This software is developed within the research project
;
;        PRISMA - Optimization methods and software for inverse problems
;                           http://www.unife.it/prisma
;
; funded by the Italian Ministry for University and Research (MIUR), under
; the PRIN2008 initiative, grant n. 2008T5KA4L, 2010-2012. This software is
; part of the package "IRMA - Image Reconstruction in Microscopy and Astronomy"
; currently under development within the PRISMA project.
;
; Version: 1.0
; Date:    July 2011
;
; Authors:
;   Roberto Cavicchioli, Marco Prato, Luca Zanni
;    Dept. of Pure Appl. Math., Univ. of Modena and Reggio Emilia, Italy
;    roberto.cavicchioli@unimore.it, marco.prato@unimore.it, luca.zanni@unimore.it
;   Mario Bertero, Patrizia Boccacci
;    DISI (Dipartimento di Informatica e Scienze dell'Informazione), University of Genova, Italy
;    bertero@disi.unige.it, boccacci@disi.unige.it
;
; Software homepage: http://www.unife.it/prisma/software
;
; Copyright (C) 2011 by M. Bertero, P. Boccacci, R. Cavicchioli, M. Prato, L. Zanni.
; ------------------------------------------------------------------------------
; COPYRIGHT NOTIFICATION
;
; Permission to copy and modify this software and its documentation for
; internal research use is granted, provided that this notice is retained
; thereon and on all copies or modifications. The authors and their
; respective Universities makes no representations as to the suitability
; and operability of this software for any purpose. It is provided "as is"
; without express or implied warranty. Use of this software for commercial
; purposes is expressly prohibited without contacting the authors.
;
; This program is free software; you can redistribute it and/or modify it
; under the terms of the GNU General Public License as published by the
; Free Software Foundation; either version 3 of the License, or (at your
; option) any later version.
;
; This program is distributed in the hope that it will be useful, but
; WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
; See the GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License along
; with this program; if not, either visite http://www.gnu.org/licenses/
; or write to
; Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
; ==============================================================================

on_error,2
if n_params() LT 1 then message, 'USE: result=magnitude(flux,[delta_t=delta_t],[surf=surf],[band=band])'

if keyword_set(band) then band_type=band else band_type='K' ;default is 'K' band
if keyword_set(delta_t) then delta_t=delta_t else delta_t=1. ;default = 1 second
if keyword_set(surf) then surf=surf else surf=1. ; default = 1 m^2

case band_type of
'K' : begin
		; band K
		lambda=2.2e-6
		width=0.4 ; in micrometers
		e0=3.9e-10
		h = 6.626d-34   ; Planck constant [Js]
		c = 3d8         ; light velocity [m/s]
		;print,flux*(h*c/lambda)/(float(delta_t)*float(surf)*e0*width)

		m=-2.5*alog10(flux*(h*c/lambda)/(float(delta_t)*float(surf)*e0*width))

		return, m
	      end
'KS' : begin
		; band K
		lambda=2.149e-6
		width=0.32 ; in micrometers
		e0=3.9e-10
		h = 6.626d-34   ; Planck constant [Js]
		c = 3d8         ; light velocity [m/s]
		;print,flux*(h*c/lambda)/(float(delta_t)*float(surf)*e0*width)

		m=-2.5*alog10(flux*(h*c/lambda)/(float(delta_t)*float(surf)*e0*width))

		return, m
	      end
endcase
end
; ==============================================================================
; End of magnitude.pro file - IRMA package
; ==============================================================================
